<?php

namespace App\Http\Controllers;

use Auth;
use Illuminate\Http\Request;
use App\Models\Message;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Mail;
use App\Events\MessageSent;
use App\Mail\TestMail;

class MessageController extends Controller
{
    public function index()
    {
        $messages = Message::paginate(20);
        return view('backend.message.index')->with('messages', $messages);
    }

    public function messageFive()
    {
        $messages = Message::whereNull('read_at')->limit(5)->get();
        return response()->json($messages);
    }

    
public function store(Request $request)
{
    $validator = Validator::make($request->all(), [
        'name' => 'required|string|max:255',
        'guardian' => 'required|string|max:255',
        'standard_id' => 'required|string|max:255',
        'email' => 'required|email',
        'phone' => 'required|string|max:20',
        'child' => 'required|string|max:255',
        'age' => 'required|numeric|min:1',
        'message' => 'nullable|string',
    ]);

    if ($validator->fails()) {
        return back()->withErrors($validator)->withInput();
    }

    $validatedData = $validator->validated();

    Message::create([
        'name' => $validatedData['name'],
        'guardian' => $validatedData['guardian'],
        'standard_id' => $validatedData['standard_id'],
        'email' => $validatedData['email'],
        'mobile' => $validatedData['phone'], // Notice: saving into 'mobile' column
        'child' => $validatedData['child'],
        'age' => $validatedData['age'],
        'message' => $validatedData['message'] ?? null,
    ]);

    $data = [
        'name' => $validatedData['name'],
        'guardian' => $validatedData['guardian'],
        'email' => $validatedData['email'],
        'phone' => $validatedData['phone'],
        'child' => $validatedData['child'],
        'age' => $validatedData['age'],
        'message' => $validatedData['message'] ?? null,
    ];

    event(new MessageSent($data));

    Mail::to("stskp96@gmail.com")->send(new TestMail($data));
    Mail::to($validatedData['email'])->send(new TestMail($data));

    return back()->with('success', 'Your message has been sent successfully!');
}

    public function show($id)
    {
        $message = Message::find($id);
        if ($message) {
            $message->read_at = now();
            $message->save();
            return view('backend.message.show')->with('message', $message);
        } else {
            return back()->with('error', 'Message not found!');
        }
    }

    public function destroy($id)
    {
        $message = Message::find($id);
        if ($message) {
            $message->delete();
            return back()->with('success', 'Deleted message successfully');
        } else {
            return back()->with('error', 'Error occurred, please try again');
        }
    }
}
