<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\NewsItem;
use App\Models\NewsCategory;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\Storage;

class NewsItemController extends Controller
{
    public function index()
    {
        $items = NewsItem::with('category')->latest()->get();
        return view('backend.news_items.index', compact('items'));
    }

    public function create()
    {
        $categories = NewsCategory::all();
        return view('backend.news_items.create', compact('categories'));
    }

    public function store(Request $request)
    {
        $request->validate([
            'news_category_id' => 'required|exists:news_categories,id',
            'title' => 'required|string|max:255',
            'short_description' => 'nullable|string|max:500',
            'description' => 'required|string',
            'image' => 'nullable|image|mimes:jpg,png,jpeg,gif|max:5120',
            'published_at' => 'nullable|date',
        ]);

        $data = $request->only([
            'news_category_id',
            'title',
            'short_description',
            'description',
            'published_at'
        ]);

        // ✅ Generate a unique slug (check if exists)
        $baseSlug = Str::slug($request->title);
        $slug = $baseSlug;
        $count = 1;
        while (NewsItem::where('slug', $slug)->exists()) {
            $slug = $baseSlug . '-' . $count++;
        }
        $data['slug'] = $slug;

        // ✅ Handle image upload
        if ($request->hasFile('image')) {
            $data['image'] = $request->file('image')->store('news_images', 'public');
        }

        NewsItem::create($data);

        return redirect()
            ->route('backend.news-items.index')
            ->with('success', 'News item created successfully.');
    }

    public function edit($id)
    {
        $item = NewsItem::findOrFail($id);
        $categories = NewsCategory::all();
        return view('backend.news_items.edit', compact('item', 'categories'));
    }

    public function update(Request $request, $id)
    {
        $item = NewsItem::findOrFail($id);

        $request->validate([
            'news_category_id' => 'required|exists:news_categories,id',
            'title' => 'required|string|max:255',
            'short_description' => 'nullable|string|max:500',
            'description' => 'required|string',
            'image' => 'nullable|image|mimes:jpg,png,jpeg,gif|max:5120',
            'published_at' => 'nullable|date',
        ]);

        $data = $request->only([
            'news_category_id',
            'title',
            'short_description',
            'description',
            'published_at'
        ]);

        // ⚠️ Do NOT regenerate slug every time — only if title has changed
        if ($item->title !== $request->title) {
            $baseSlug = Str::slug($request->title);
            $slug = $baseSlug;
            $count = 1;
            while (NewsItem::where('slug', $slug)->where('id', '!=', $item->id)->exists()) {
                $slug = $baseSlug . '-' . $count++;
            }
            $data['slug'] = $slug;
        }

        // ✅ Handle image upload
        if ($request->hasFile('image')) {
            if ($item->image && Storage::disk('public')->exists($item->image)) {
                Storage::disk('public')->delete($item->image);
            }
            $data['image'] = $request->file('image')->store('news_images', 'public');
        }

        $item->update($data);

        return redirect()
            ->route('backend.news-items.index')
            ->with('success', 'News item updated successfully.');
    }

    public function destroy($id)
    {
        $item = NewsItem::findOrFail($id);

        if ($item->image && Storage::disk('public')->exists($item->image)) {
            Storage::disk('public')->delete($item->image);
        }

        $item->delete();

        return redirect()
            ->route('backend.news-items.index')
            ->with('success', 'News item deleted successfully.');
    }
}
