<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Notice;
use App\Models\NoticeCategory;
use Illuminate\Support\Facades\Storage;

class NoticeController extends Controller
{
    public function index()
    {
        $notices = Notice::with('category')->latest()->get();
        return view('backend.notices.index', compact('notices'));
    }

    public function create()
    {
        $categories = NoticeCategory::all();
        return view('backend.notices.create', compact('categories'));
    }

    public function store(Request $request)
    {
        $request->validate([
            'notice_category_id' => 'required|exists:notice_categories,id',
            'title' => 'required|string|max:255',
            'date' => 'required|date',
            'description' => 'required|string',
            'pdf_path' => 'nullable|file|mimes:pdf|max:5120',
        ]);

        $data = $request->only(['notice_category_id', 'title', 'date', 'description']);

        if ($request->hasFile('pdf_path')) {
            $data['pdf_path'] = $request->file('pdf_path')->store('notices', 'public');
        }

        Notice::create($data);

        return redirect()->route('backend.notices.index')->with('success', 'Notice created successfully.');
    }

    public function edit($id)
    {
        $notice = Notice::findOrFail($id);
        $categories = NoticeCategory::all();
        return view('backend.notices.edit', compact('notice', 'categories'));
    }

    public function update(Request $request, $id)
    {
        $notice = Notice::findOrFail($id);

        $request->validate([
            'notice_category_id' => 'required|exists:notice_categories,id',
            'title' => 'required|string|max:255',
            'date' => 'required|date',
            'description' => 'required|string',
            'pdf_path' => 'nullable|file|mimes:pdf|max:5120',
        ]);

        $data = $request->only(['notice_category_id', 'title', 'date', 'description']);

        if ($request->hasFile('pdf_path')) {
            if ($notice->pdf_path && Storage::disk('public')->exists($notice->pdf_path)) {
                Storage::disk('public')->delete($notice->pdf_path);
            }
            $data['pdf_path'] = $request->file('pdf_path')->store('notices', 'public');
        }

        $notice->update($data);

        return redirect()->route('backend.notices.index')->with('success', 'Notice updated successfully.');
    }

    public function destroy($id)
    {
        $notice = Notice::findOrFail($id);

        if ($notice->pdf_path && Storage::disk('public')->exists($notice->pdf_path)) {
            Storage::disk('public')->delete($notice->pdf_path);
        }

        $notice->delete();

        return redirect()->route('backend.notices.index')->with('success', 'Notice deleted successfully.');
    }

      public function categoryWise($slug)
    {
        $category = NoticeCategory::where('slug', $slug)->firstOrFail();
        $notices = Notice::where('notice_category_id', $category->id)
                         ->orderBy('date', 'desc')
                         ->get();

        return view('frontend.pages.notice-list', compact('category', 'notices'));
    }
}
