<?php

namespace App\Http\Controllers;

use App\Models\Testimonial;
use Illuminate\Http\Request;

class TestimonialController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        $testimonials=Testimonial::latest()->paginate('10');
        return view('backend.testimonials.index',compact('testimonials'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        return view('backend.testimonials.create');
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
         // ✅ Validation
         $request->validate([
            'title' => 'nullable|string|max:255',
            'image' => 'nullable|image|mimes:jpg,png,jpeg,gif',
            'profiles' => 'nullable|string',
            'description' => 'nullable|string',
            'review' => 'nullable|integer|between:1,5',
        ]);
    
        // ✅ Handle image upload
        $imagePath = null;
        if ($request->hasFile('image')) {
            $image = $request->file('image');
            $extension = $image->getClientOriginalExtension();
            $imageName = time() . '_image.' . $extension;
            $image->move(public_path('admin/img/'), $imageName);
            $imagePath = 'admin/img/' . $imageName;
        }
    
        // ✅ Create new record
        Testimonial::create([
            'title' => $request->title,
            'profiles' => $request->profiles,
            'description' => $request->description,
            'review' => $request->review,
            'image' => $imagePath,
            'status' => $request->status ? 1 : 0,
        ]);
    
        // ✅ Redirect with success message
        return redirect()->route('backend.testimonials.index')
                         ->with('success', 'Testimonial created successfully.');
    }

    /**
     * Display the specified resource.
     */
    public function show(Testimonial $testimonial)
    {
        //
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(Testimonial $testimonial)
    {
        return view('backend.testimonials.edit',compact('testimonial'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, Testimonial $testimonial)
    {
        $request->validate([
            'title' => 'nullable|string|max:255',
            'image' => 'nullable|image|mimes:jpg,png,jpeg,gif',
            'profiles' => 'nullable|string',
            'description' => 'nullable|string',
            'review' => 'nullable|integer|between:1,5',
        ]);

        if ($request->hasFile('image')) {
            // Delete old image if exists
            if ($testimonial->image && file_exists(public_path($testimonial->image))) {
                unlink(public_path($testimonial->image));
            }

            // Upload new image
            $image = $request->file('image');
            $imageName = time() . '.' . $image->getClientOriginalExtension();
            $image->move(public_path('admin/img'), $imageName);
            $testimonial->image = 'admin/img/' . $imageName; // Save relative path
        }

        $testimonial->update([
            'title' => $request->title,
            'profiles' => $request->profiles,
            'description' => $request->description,
            'review' => $request->review,
            'status' => $request->status ? 1 : 0,
        ]);

        return redirect()->route('backend.testimonials.index')->with('success', 'Testimonials updated successfully.');
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(Testimonial $testimonial)
    {
        if ($testimonial->image && file_exists(public_path($testimonial->image))) {
            unlink(public_path($testimonial->image));
        }

        $testimonial->delete();
        return redirect()->route('backend.testimonials.index')->with('success', 'Testimonials deleted successfully.');
    }
}
